#!/usr/bin/perl

use strict;
use warnings;
use utf8;
use Encode;
use Encode::Guess qw/euc-jp shiftjis iso-2022-jp/;
use Tk;
use Lingua::JA::Fold qw(fold);
use open IN  => ":bytes"; # 入力される文字コードが混在している可能性があるので

# 設定いろいろ ( IPAフォント使用時に合わせてあります)
my $font_family            = 'IPA明朝';  # テキスト表示画面のフォント名
my $font_size              = 11;         # テキスト表示画面のフォントサイズ
my $font_style             = 'normal';   # テキスト表示画面のフォントスタイル
my $font_color             = '#000000';  # テキスト表示画面のフォントカラー
my $background_color       = '#e0ffff';  # テキスト表示画面の背景色
my $text_width             = 523;        # テキスト表示開始部の幅
my $text_height            = 28;         # テキスト表示開始部の高さ
my $line_horizontal_margin = 26;         # 各行間の横の余裕
my $chr_vertical_margin    = 15;         # 各文字間の縦の余裕
my $fold_length            = 42;         # 文庫本 1ページの一般的な一行の文字数
my $page_lines             = 18;         # 文庫本 1ページの一般的な行数
my $counting_font_family   = 'IPAゴシック'; # ページ数表示部のフォント名
my $counting_font_style    = 'bold';     # ページ数表示部のフォントスタイル
my $shiori_file            = 'pochitate_shiori.txt'; # 栞ファイルの場所

my @chr_ids;             # ページ内各文字の ID の配列
my @offsets = (0);       # 既読ページのテキストデータ読み出し後のオフセット配列
my $page = 0;            # 現在表示しているページ数
my $shiori_page;         # 栞を挟んだページ
my $filename;            # 読み込むテキストファイル名

# 引き数が無くても栞があったら自動復帰
unless ($filename = $ARGV[0]){
    if (-e $shiori_file){
        open my $sh,"<", $shiori_file
            or die "Couldn't open $shiori_file for reading: $!\n";
        $filename = <$sh>;
        close $sh;
        chomp $filename;
        ($filename, $offsets[0], $shiori_page) = split ' ', $filename;
        ($ARGV[0], $page) = ($filename, $shiori_page);
    } else {
        die "There is no argument.\n";
    }
}

# メインウィンドウの準備
my $top = MainWindow->new( -title => 'ぽちたて 0.0.3' );

# フレームの準備
my $frame_top = $top->Frame();
my $frame_bottom = $top->Frame();

# 主画面となるキャンバスウィジェットの配置
$frame_top = $top->Canvas( width => 603, height => 671 );
$frame_top->create( 'rectangle', 1, 1, 602, 670, -fill => $background_color );

# ページ数表示部、ページ移動ボタンの配置
# (「＜」で次ページ、「＞」で前ページ、「×」で終了)
$frame_bottom->Label( -textvariable => \$page,
          -font => [ $counting_font_family, 12, $counting_font_style ] )
          ->pack( -side => 'left' );
$frame_bottom->Button( -text => '＜',
          -command => [ \&next_tategaki, \@chr_ids, \@offsets, \$page ],
          -font => [ $font_family, 4, $font_style ] )->pack( -side => 'left' );
$frame_bottom->Button( -text => '＞',
          -command => [ \&prev_tategaki, \@chr_ids, \@offsets, \$page ],
          -font => [ $font_family, 4, $font_style ] )->pack( -side => 'left' );
$frame_bottom->Button( -text => '栞',
          -command => \&shiori,
          -font => [ $font_family, 10, $font_style ] )->pack( -side => 'left' );
$frame_bottom->Button( -text => '×', -command => \&exit,
          -font => [ $font_family, 4, $font_style ] )->pack( -side => 'left' );

$top->bind('<space>' => \&next_ward); # キー操作 ' ' で次ページ
$top->bind('<p>' => \&prev_ward);     # キー操作 'p' で前ページ
$top->bind('<b>' => \&shiori);        # キー操作 'b' で栞を挟む
$top->bind('<q>' => \&exit);          # キー操作 'q' で終了

# フレーム内の配置
$frame_top->pack();
$frame_bottom->pack( -side => 'right' );

# テキストファイルからの読み込みと文字コード判定・折り返し処理
my $strs;
for (@ARGV){
    open IN, $_ or die "Couldn't open IN for reading: $!\n";
    my $content .= join '', <IN>;
    close IN;
    my $guess = guess_encoding($content);
    ref $guess or die "Couldn't guess: $guess\n";
    open IN, $_ or die "Couldn't open IN for reading: $!\n";
    while (<IN>){
        $_ = $guess->decode($_);
        $strs .= fold( 'text' => $_, 'length' => $fold_length,
                       'mode' => 'traditional' );
    }
    close IN;
}

open my $fh, "<", \$strs or die 'Couldn\'t open $fh for reading: ', "$!\n";
seek $fh, $offsets[0], 0;
&next_tategaki(\@chr_ids, \@offsets, \$page);

MainLoop();

# 次ページ表示
sub next_tategaki{
    my ( $chr_ids_ref, $offsets_ref, $page_ref) = @_; 
    return if eof $fh;
    for (@$chr_ids_ref){
        $frame_top->delete( $_ );
    }
    @$chr_ids_ref = ();
    &write_tategaki($chr_ids_ref);
    push @$offsets_ref, tell $fh or die "Couldn't tell the file offset: $!\n";
    $$page_ref++;
    undef;
}

# 前ページ表示
sub prev_tategaki{
    my ( $chr_ids_ref, $offsets_ref, $page_ref) = @_;
    return if $#offsets < 0 || defined $shiori_page && $shiori_page > $page - 2;
    for (@$chr_ids_ref){
        $frame_top->delete( $_ );
    }
    @$chr_ids_ref = ();
    pop @$offsets_ref if $#offsets > 1;
    seek $fh, $offsets[-2], 0;
    &write_tategaki($chr_ids_ref);
    $$page_ref-- if $$page_ref > 1;
    undef;
}

# 縦書き描画
sub write_tategaki{
    my $chr_ids_ref = shift;
    my $lines;
    my $line_width = $text_width;
    while (<$fh>){
        $_ = decode_utf8($_);
        my $chr_height = $text_height;
        my @string = split(//, $_);
        for (@string){
            my $id = $frame_top->create( 'text', $line_width, $chr_height,
                          -fill => $font_color, -text => $_,
                          -font => [ $font_family, $font_size, $font_style ] );
            push @$chr_ids_ref, $id;
            $chr_height += $chr_vertical_margin;
        }
        $line_width -= $line_horizontal_margin;
        $lines++;
        last if ($lines > $page_lines - 1);
    }
    undef;
}

# 栞を挟む
sub shiori{
    open my $sh, ">", $shiori_file
        or die "Couldn't open $shiori_file for writing: $!\n";
    print $sh $filename, ' ', $offsets[-2], ' ' , $page - 1, "\n";
    close $sh or die "Couldn't close $shiori_file: $!\n";
    undef;
}

sub next_ward{
    &next_tategaki(\@chr_ids, \@offsets, \$page);
    undef;
}

sub prev_ward{
    &prev_tategaki(\@chr_ids, \@offsets, \$page);
    undef;
}

# Copyright (c) 2009, 犬山ぽち丸　
# このコードは、Perl自体と同じライセンスで配布します。
# Copyright (c) 2009, Pochimaru Inuyama. All rights reserved.
# This code is distributed under the same terms as Perl itself.
